/*
  ==============================================================================

    SonicCrypt Chaos Engine
    Copyright (C) 2025 Sebastian Sünkler

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

  ==============================================================================
*/

#pragma once

#include <JuceHeader.h>
#include "PluginProcessor.h"

class AboutOverlay : public juce::Component {
public:
    AboutOverlay() {
        // OK Button
        okButton.setButtonText("OK");
        okButton.setColour(juce::TextButton::buttonColourId, juce::Colours::transparentBlack);
        okButton.onClick = [this] { setVisible(false); };
        addAndMakeVisible(okButton);

        // AGPL Lizenz Link
        licenseLink.setButtonText("GNU AGPL v3 License");
        licenseLink.setURL(juce::URL("https://www.gnu.org/licenses/agpl-3.0.html.en"));
        // Hier nutzen wir deine Chaos Engine Akzentfarbe (0xff008B9E) statt dem Blau des Sequencers
        licenseLink.setColour(juce::HyperlinkButton::textColourId, juce::Colour(0xff008B9E));
        addAndMakeVisible(licenseLink);

        // JUCE Link ("Made with JUCE")
        juceLink.setButtonText("Made with JUCE");
        juceLink.setURL(juce::URL("https://juce.com"));
        juceLink.setColour(juce::HyperlinkButton::textColourId, juce::Colours::white.withAlpha(0.6f));
        addAndMakeVisible(juceLink);
    }

    void paint(juce::Graphics& g) override {
        g.fillAll(juce::Colours::black.withAlpha(0.7f));

        // Layout-Berechnung (identisch zum Sequencer)
        auto area = getLocalBounds().toFloat().withSizeKeepingCentre(500, 280);

        // Hintergrund
        g.setColour(juce::Colour(0xff1E1E1E).darker(0.2f));
        g.fillRoundedRectangle(area, 10.0f);
        g.setColour(juce::Colours::black);
        g.drawRoundedRectangle(area, 10.0f, 2.0f);

        auto content = area.reduced(20);

        // Icon (Chaos Engine Farbe)
        auto iconArea = content.removeFromLeft(80).removeFromTop(80);
        iconArea.setY(iconArea.getY() + 20);
        g.setColour(juce::Colour(0xff008B9E));
        g.fillEllipse(iconArea);
        g.setColour(juce::Colours::black.withAlpha(0.3f));
        g.setFont(juce::FontOptions(50.0f).withStyle("Bold"));
        g.drawText("i", iconArea.toNearestInt(), juce::Justification::centred);

        content.removeFromLeft(20);

        // Überschrift
        g.setColour(juce::Colours::white);
        g.setFont(juce::FontOptions(20.0f).withStyle("Bold"));
        g.drawText("About SonicCrypt Chaos Engine", content.removeFromTop(30).toNearestInt(), juce::Justification::topLeft);

        // Entwickler
        g.setFont(juce::FontOptions(15.0f));
        g.drawText(juce::CharPointer_UTF8("Idea & Development: Sebastian S\xc3\xbc\x6ekler"), content.removeFromTop(25).toNearestInt(), juce::Justification::topLeft);

        // Beschreibung (Chaos Engine spezifisch)
        g.setColour(juce::Colours::grey);
        g.setFont(juce::FontOptions(13.0f));
        g.drawFittedText("An experimental industrial noise synthesizer with smart chaos generation.\n\nVersion 1.0", content.removeFromTop(50).toNearestInt(), juce::Justification::topLeft, 3);

        // Label "Licensed under:"
        g.drawText("Licensed under:", content.removeFromTop(20).toNearestInt(), juce::Justification::topLeft);

        // --- Steinberg Trademark Hinweis (Footer) ---
        auto footerArea = area.removeFromBottom(20).reduced(10, 0);
        g.setColour(juce::Colours::grey.withAlpha(0.5f));
        g.setFont(10.0f);
        g.drawFittedText("VST is a registered trademark of Steinberg Media Technologies GmbH.", footerArea.toNearestInt(), juce::Justification::centred, 1);
    }

    void resized() override {
        auto area = getLocalBounds().withSizeKeepingCentre(500, 280);

        // OK Button (angepasste Position analog zum Sequencer)
        okButton.setBounds(area.getX() + 350, area.getBottom() - 45, 80, 25);

        // Lizenz Link
        licenseLink.setBounds(area.getX() + 120, area.getY() + 145, 150, 20); // Leicht angepasst wegen mehr Textzeilen

        // JUCE Link 
        juceLink.setBounds(area.getX() + 20, area.getBottom() - 45, 120, 25);
    }

private:
    juce::TextButton okButton;
    juce::HyperlinkButton licenseLink;
    juce::HyperlinkButton juceLink;
};

class LoadingOverlay : public juce::Component {
public: LoadingOverlay() {} void paint(juce::Graphics& g) override { g.fillAll(juce::Colours::black.withAlpha(0.85f)); g.setColour(juce::Colour::fromString("FF00BFFF")); g.setFont(juce::FontOptions(40.0f).withStyle("Bold")); float alpha = 0.5f + 0.5f * std::sin(juce::Time::getMillisecondCounter() / 150.0); g.setColour(juce::Colour::fromString("FF00BFFF").withAlpha(alpha)); g.drawText("GENERATING CHAOS...", getLocalBounds(), juce::Justification::centred); }
};

class PresetListComponent : public juce::ListBoxModel {
public: PresetListComponent(juce::ListBox& lb, std::function<void(juce::File)> onSelect) : listBox(lb), onSelection(onSelect) {} int getNumRows() override { return presets.size(); } void paintListBoxItem(int rowNumber, juce::Graphics& g, int width, int height, bool rowIsSelected) override { if (rowIsSelected) g.fillAll(juce::Colour::fromString("FF00BFFF").withAlpha(0.2f)); g.setColour(rowIsSelected ? juce::Colour::fromString("FF00BFFF") : juce::Colours::white); g.setFont(14.0f); g.drawText(presets[rowNumber].getFileNameWithoutExtension(), 5, 0, width, height, juce::Justification::centredLeft); } void listBoxItemClicked(int row, const juce::MouseEvent&) override { if (onSelection) onSelection(presets[row]); } void refresh(const juce::File& folder) { presets.clear(); if (folder.isDirectory()) for (auto entry : juce::RangedDirectoryIterator(folder, false, "*.scp")) presets.add(entry.getFile()); listBox.updateContent(); }
private: juce::ListBox& listBox; juce::Array<juce::File> presets; std::function<void(juce::File)> onSelection;
};

// --- CHAOS ORB BUTTON ---
class ChaosPulseButton : public juce::Component {
public:
    std::function<void()> onClick;

    void setLevel(float level) {
        currentLevel = currentLevel * 0.8f + level * 0.2f; // Smooth
        repaint();
    }

    void mouseDown(const juce::MouseEvent&) override { isDown = true; repaint(); }
    void mouseUp(const juce::MouseEvent&) override { isDown = false; if (onClick) onClick(); repaint(); }

    void paint(juce::Graphics& g) override {
        auto bounds = getLocalBounds().toFloat();
        auto center = bounds.getCentre();
        float maxRadius = juce::jmin(bounds.getWidth(), bounds.getHeight()) / 2.0f;
        float baseRadius = maxRadius * 0.8f;
        if (isDown) baseRadius *= 0.9f;

        g.setColour(juce::Colour::fromString("FF00BFFF"));

        if (currentLevel > 0.01f) {
            juce::Path chaosPath;
            int points = 60;
            for (int i = 0; i <= points; ++i) {
                float angle = juce::MathConstants<float>::twoPi * i / points;
                float noise = rng.nextFloat() * 2.0f - 1.0f;
                float distortion = currentLevel * 20.0f * noise;
                float r = baseRadius + distortion;
                angle += (float)juce::Time::getMillisecondCounter() / 1000.0f;
                auto p = center.getPointOnCircumference(r, angle);
                if (i == 0) chaosPath.startNewSubPath(p); else chaosPath.lineTo(p);
            }
            chaosPath.closeSubPath();
            g.setOpacity(0.8f + currentLevel * 0.2f);
            g.fillPath(chaosPath);
            g.setColour(juce::Colours::white.withAlpha(0.8f));
            g.strokePath(chaosPath, juce::PathStrokeType(2.0f));
        }
        else {
            g.fillEllipse(center.getX() - baseRadius, center.getY() - baseRadius, baseRadius * 2, baseRadius * 2);
        }

        g.setColour(juce::Colours::black);
        g.setFont(juce::FontOptions(20.0f).withStyle("Bold"));
        g.drawText("GENERATE\nCHAOS", getLocalBounds(), juce::Justification::centred);
    }

private:
    float currentLevel = 0.0f;
    bool isDown = false;
    juce::Random rng;
};

class SonicCryptChaosAudioProcessorEditor : public juce::AudioProcessorEditor, public juce::Button::Listener, public juce::Timer {
public:
    SonicCryptChaosAudioProcessorEditor(SonicCryptChaosAudioProcessor&);
    ~SonicCryptChaosAudioProcessorEditor() override;
    void paint(juce::Graphics&) override;
    void resized() override;
    void buttonClicked(juce::Button*) override;
    void timerCallback() override;

private:
    SonicCryptChaosAudioProcessor& audioProcessor;
    const int BASE_WIDTH = 1000; const int BASE_HEIGHT = 600;

    bool constructorFinished = false;

    LoadingOverlay overlay; AboutOverlay aboutOverlay;

    juce::GroupComponent presetGroup{ "presets", "PRESETS" }; juce::ListBox presetListBox; std::unique_ptr<PresetListComponent> presetListModel;
    juce::TextButton savePresetButton{ "SAVE NEW" }, refreshButton{ "REFRESH" };

    juce::ImageButton logoButton; juce::TextButton copyrightBtn{ "copyright" };

    // ORB BUTTON REPLACES TEXT BUTTON
    ChaosPulseButton chaosButton;
    juce::TextButton loadFolderButton{ "LOAD FOLDER" };
    juce::Label folderPathLabel;

    juce::ComboBox modeSelector; std::unique_ptr<juce::AudioProcessorValueTreeState::ComboBoxAttachment> modeAttachment;
    juce::Label typeLabel;

    juce::ComboBox rhythmRateSelector; std::unique_ptr<juce::AudioProcessorValueTreeState::ComboBoxAttachment> rhythmRateAttachment;
    juce::Label rhythmRateLabel;

    juce::GroupComponent lfoGroup{ "lfoGroup", "DRIFT LFO" }; juce::ToggleButton lfoActiveButton{ "Active" }; std::unique_ptr<juce::AudioProcessorValueTreeState::ButtonAttachment> lfoActiveAttachment;
    juce::ComboBox lfoRateSelector; std::unique_ptr<juce::AudioProcessorValueTreeState::ComboBoxAttachment> lfoRateAttachment; juce::Label lfoRateLabel;
    juce::Slider lfoDepthSlider; std::unique_ptr<juce::AudioProcessorValueTreeState::SliderAttachment> lfoDepthAttachment; juce::Label lfoDepthLabel;
    using SliderAttachment = juce::AudioProcessorValueTreeState::SliderAttachment;
    juce::Slider driveSlider, filterSlider, resSlider, reverbSlider, delaySlider; juce::Label driveLabel, filterLabel, resLabel, reverbLabel, delayLabel;
    std::unique_ptr<SliderAttachment> driveAttachment, filterAttachment, resAttachment, reverbAttachment, delayAttachment;
    juce::Image logoImage; std::shared_ptr<juce::FileChooser> fileChooser;
    void setupSlider(juce::Slider&, juce::Label&, std::unique_ptr<SliderAttachment>&, juce::String, juce::String);
    void showAboutWindow();
    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR(SonicCryptChaosAudioProcessorEditor)
};